import { BinaryParser } from './serdes/binary-parser';
import { AccountID } from './types/account-id';
import { BinarySerializer, BytesList } from './serdes/binary-serializer';
import { sha512Half, transactionID } from './hashes';
import { type XrplDefinitionsBase } from './enums';
import { JsonObject } from './types/serialized-type';
/**
 * Construct a BinaryParser
 *
 * @param bytes hex-string or Uint8Array to construct BinaryParser from
 * @param definitions rippled definitions used to parse the values of transaction types and such.
 *                          Can be customized for sidechains and amendments.
 * @returns BinaryParser
 */
declare const makeParser: (bytes: string | Uint8Array, definitions?: XrplDefinitionsBase) => BinaryParser;
/**
 * Parse BinaryParser into JSON
 *
 * @param parser BinaryParser object
 * @param definitions rippled definitions used to parse the values of transaction types and such.
 *                          Can be customized for sidechains and amendments.
 * @returns JSON for the bytes in the BinaryParser
 */
declare const readJSON: (parser: BinaryParser, definitions?: XrplDefinitionsBase) => JsonObject;
/**
 * Parse a hex-string into its JSON interpretation
 *
 * @param bytes hex-string to parse into JSON
 * @param definitions rippled definitions used to parse the values of transaction types and such.
 *                          Can be customized for sidechains and amendments.
 * @returns JSON
 */
declare const binaryToJSON: (bytes: string, definitions?: XrplDefinitionsBase) => JsonObject;
/**
 * Interface for passing parameters to SerializeObject
 *
 * @field set signingFieldOnly to true if you want to serialize only signing fields
 */
interface OptionObject {
    prefix?: Uint8Array;
    suffix?: Uint8Array;
    signingFieldsOnly?: boolean;
    definitions?: XrplDefinitionsBase;
}
/**
 * Function to serialize JSON object representing a transaction
 *
 * @param object JSON object to serialize
 * @param opts options for serializing, including optional prefix, suffix, signingFieldOnly, and definitions
 * @returns A Uint8Array containing the serialized object
 */
declare function serializeObject(object: JsonObject, opts?: OptionObject): Uint8Array;
/**
 * Serialize an object for signing
 *
 * @param transaction Transaction to serialize
 * @param prefix Prefix bytes to put before the serialized object
 * @param opts.definitions Custom rippled types to use instead of the default. Used for sidechains and amendments.
 * @returns A Uint8Array with the serialized object
 */
declare function signingData(transaction: JsonObject, prefix?: Uint8Array, opts?: {
    definitions?: XrplDefinitionsBase;
}): Uint8Array;
/**
 * Interface describing fields required for a Claim
 */
interface ClaimObject extends JsonObject {
    channel: string;
    amount: string | number;
}
/**
 * Serialize a signingClaim
 *
 * @param claim A claim object to serialize
 * @param opts.definitions Custom rippled types to use instead of the default. Used for sidechains and amendments.
 * @returns the serialized object with appropriate prefix
 */
declare function signingClaimData(claim: ClaimObject): Uint8Array;
/**
 * Serialize a transaction object for multiSigning
 *
 * @param transaction transaction to serialize
 * @param signingAccount Account to sign the transaction with
 * @param opts.definitions Custom rippled types to use instead of the default. Used for sidechains and amendments.
 * @returns serialized transaction with appropriate prefix and suffix
 */
declare function multiSigningData(transaction: JsonObject, signingAccount: string | AccountID, opts?: {
    definitions: XrplDefinitionsBase;
}): Uint8Array;
/**
 * Interface describing fields required for a Batch signer
 * @property flags - Flags indicating Batch transaction properties
 * @property txIDs - Array of transaction IDs included in the Batch
 */
interface BatchObject extends JsonObject {
    flags: number;
    txIDs: string[];
}
/**
 * Serialize a signingClaim
 *
 * @param batch A Batch object to serialize.
 * @returns the serialized object with appropriate prefix
 */
declare function signingBatchData(batch: BatchObject): Uint8Array;
export { BinaryParser, BinarySerializer, BytesList, ClaimObject, BatchObject, makeParser, serializeObject, readJSON, multiSigningData, signingData, signingClaimData, binaryToJSON, sha512Half, transactionID, signingBatchData, };
