"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Issue = void 0;
const utils_1 = require("@xrplf/isomorphic/utils");
const binary_parser_1 = require("../serdes/binary-parser");
const account_id_1 = require("./account-id");
const currency_1 = require("./currency");
const serialized_type_1 = require("./serialized-type");
const hash_192_1 = require("./hash-192");
/**
 * Type guard for Issue Object
 */
function isIssueObject(arg) {
    const keys = Object.keys(arg).sort();
    const isXRP = keys.length === 1 && keys[0] === 'currency';
    const isIOU = keys.length === 2 && keys[0] === 'currency' && keys[1] === 'issuer';
    const isMPT = keys.length === 1 && keys[0] === 'mpt_issuance_id';
    return isXRP || isIOU || isMPT;
}
/**
 * Class for serializing/Deserializing Amounts
 */
class Issue extends serialized_type_1.SerializedType {
    constructor(bytes) {
        super(bytes !== null && bytes !== void 0 ? bytes : Issue.ZERO_ISSUED_CURRENCY.bytes);
    }
    /**
     * Construct an amount from an IOU or string amount
     *
     * @param value An Amount, object representing an IOU, MPTAmount, or a string
     *     representing an integer amount
     * @returns An Issue object
     */
    static from(value) {
        if (value instanceof Issue) {
            return value;
        }
        if (isIssueObject(value)) {
            if (value.currency) {
                const currency = currency_1.Currency.from(value.currency.toString()).toBytes();
                //IOU case
                if (value.issuer) {
                    const issuer = account_id_1.AccountID.from(value.issuer.toString()).toBytes();
                    return new Issue((0, utils_1.concat)([currency, issuer]));
                }
                //XRP case
                return new Issue(currency);
            }
            // MPT case
            if (value.mpt_issuance_id) {
                const mptIssuanceIdBytes = hash_192_1.Hash192.from(value.mpt_issuance_id.toString()).toBytes();
                return new Issue(mptIssuanceIdBytes);
            }
        }
        throw new Error('Invalid type to construct an Amount');
    }
    /**
     * Read an amount from a BinaryParser
     *
     * @param parser BinaryParser to read the Amount from
     * @param hint The number of bytes to consume from the parser.
     * For an MPT amount, pass 24 (the fixed length for Hash192).
     *
     * @returns An Issue object
     */
    static fromParser(parser, hint) {
        if (hint === hash_192_1.Hash192.width) {
            const mptBytes = parser.read(hash_192_1.Hash192.width);
            return new Issue(mptBytes);
        }
        const currency = parser.read(20);
        if (new currency_1.Currency(currency).toJSON() === 'XRP') {
            return new Issue(currency);
        }
        const currencyAndIssuer = [currency, parser.read(20)];
        return new Issue((0, utils_1.concat)(currencyAndIssuer));
    }
    /**
     * Get the JSON representation of this Amount
     *
     * @returns the JSON interpretation of this.bytes
     */
    toJSON() {
        // If the buffer is exactly 24 bytes, treat it as an MPT amount.
        if (this.toBytes().length === hash_192_1.Hash192.width) {
            return {
                mpt_issuance_id: this.toHex().toUpperCase(),
            };
        }
        const parser = new binary_parser_1.BinaryParser(this.toString());
        const currency = currency_1.Currency.fromParser(parser);
        if (currency.toJSON() === 'XRP') {
            return { currency: currency.toJSON() };
        }
        const issuer = account_id_1.AccountID.fromParser(parser);
        return {
            currency: currency.toJSON(),
            issuer: issuer.toJSON(),
        };
    }
}
exports.Issue = Issue;
Issue.ZERO_ISSUED_CURRENCY = new Issue(new Uint8Array(20));
//# sourceMappingURL=issue.js.map