import { BytesList } from '../serdes/binary-serializer';
import { BinaryParser } from '../serdes/binary-parser';
import { XrplDefinitionsBase } from '../enums';
type JSON = string | number | boolean | null | undefined | JSON[] | JsonObject;
type JsonObject = {
    [key: string]: JSON;
};
/**
 * The base class for all binary-codec types
 */
declare class SerializedType {
    protected readonly bytes: Uint8Array;
    constructor(bytes?: Uint8Array);
    static fromParser(parser: BinaryParser, hint?: number): SerializedType;
    static from(value: SerializedType | JSON | bigint): SerializedType;
    /**
     * Write the bytes representation of a SerializedType to a BytesList
     *
     * @param list The BytesList to write SerializedType bytes to
     */
    toBytesSink(list: BytesList): void;
    /**
     * Get the hex representation of a SerializedType's bytes
     *
     * @returns hex String of this.bytes
     */
    toHex(): string;
    /**
     * Get the bytes representation of a SerializedType
     *
     * @returns A Uint8Array of the bytes
     */
    toBytes(): Uint8Array;
    /**
     * Return the JSON representation of a SerializedType
     *
     * @param _definitions rippled definitions used to parse the values of transaction types and such.
     *                          Unused in default, but used in STObject, STArray
     *                          Can be customized for sidechains and amendments.
     * @returns any type, if not overloaded returns hexString representation of bytes
     */
    toJSON(_definitions?: XrplDefinitionsBase, _fieldName?: string): JSON;
    /**
     * @returns hexString representation of this.bytes
     */
    toString(): string;
}
/**
 * Base class for SerializedTypes that are comparable.
 *
 * @template T - What types you want to allow comparisons between. You must specify all types. Primarily used to allow
 * comparisons between built-in types (like `string`) and SerializedType subclasses (like `Hash`).
 *
 * Ex. `class Hash extends Comparable<Hash | string>`
 */
declare class Comparable<T extends Object> extends SerializedType {
    lt(other: T): boolean;
    eq(other: T): boolean;
    gt(other: T): boolean;
    gte(other: T): boolean;
    lte(other: T): boolean;
    /**
     * Overload this method to define how two Comparable SerializedTypes are compared
     *
     * @param other The comparable object to compare this to
     * @returns A number denoting the relationship of this and other
     */
    compareTo(other: T): number;
}
export { SerializedType, Comparable, JSON, JsonObject };
